# libbuild2-qt

This module provides compilation support for the Qt `moc`, `rcc`, and `uic`
compilers.

## Usage overview

This module is part of the standard pre-installed `build2` modules and no
extra integration steps are required other than the `using` directive(s) in
your `buildfile`.

A specific compiler submodule can be loaded directly:

```
using qt.moc
using qt.rcc
```

Otherwise, if all three compilers are going to be used, the `qt` module
automatically loads all of them:

```
using qt
```

By default, the module assumes Qt6 but a different version can be specified
before loading the (sub)modules, for example:

```
qt.version = 5

using qt
```

Note that this module does not depend on any Qt packages so user projects are
expected to add the required Qt dependencies to their `manifest`. For example:

```
depends: * Qt6Moc ^6.4.3
depends: * Qt6Uic ^6.4.3
depends: * Qt6Rcc ^6.4.3
depends: libQt6Core ^6.4.3
depends: libQt6Gui ^6.4.3
depends: libQt6Widgets ^6.4.3
```

The following simplified example shows how the Qt compilers can be used in a
build. In this `buildfile` C++ source files produced by `moc`, `rcc`, and
`uic` are incorporated into the build of the `hello` executable:

```
using qt

import libs  = libQt6Core%lib{Qt6Core}
import libs += libQt6Gui%lib{Qt6Gui}
import libs += libQt6Widgets%lib{Qt6Widgets}

# Dependencies on the primary source files.
#
exe{hello}: {hxx cxx}{hello}

# Dependencies on the outputs of the Qt compilers.
#
exe{hello}: hxx{ui_hello} moc{hello} cxx{qrc_hello}

# Dependencies on the libraries.
#
exe{hello}: $libs

# Dependencies of the Qt compiler outputs on their input files.
#
moc{hello}:     cxx{hello} # Compile hello.cxx with moc.
cxx{qrc_hello}: qrc{hello} # Compile hello.qrc with rcc.
hxx{ui_hello}:  ui{hello}  # Compile hello.ui with uic.
```

Note that in this document we use the `.hxx`/`.cxx` C++ header/source file
extensions but any other extensions can be used as well.


## `moc` module

The `moc` module runs `moc` (the Qt Meta-object Compiler) on C++ header or
source files.

Loading the `moc` module:

```
using qt.moc
```

### `moc` configuration variables

```
[strings] qt.moc.options             ?= [null]
[bool]    qt.moc.auto_preprocessor   ?= true
[bool]    qt.moc.auto_poptions       ?= $qt.moc.auto_preprocessor
[bool]    qt.moc.auto_predefs        ?= $qt.moc.auto_preprocessor
[bool]    qt.moc.auto_sys_hdr_dirs   ?= $qt.moc.auto_preprocessor
[bool]    qt.moc.include_with_quotes ?= false
```

* `qt.moc.options`

  Options that will be passed directly to `moc`. Default value is `null`.

* `qt.moc.auto_preprocessor`

  Fallback variable used if any of the other `qt.moc.auto_*` variables are
  `null` or undefined. Default value is `true`.

* `qt.moc.auto_poptions`

  If `true`, automatically pass the project's preprocessor options
  (`cc.poptions` and `cxx.poptions`) to `moc`. Default value is the value of
  `qt.moc.auto_preprocessor` (which is `true` by default).

* `qt.moc.auto_predefs`

  If `true`, automatically pass the C++ compiler's pre-defined macros to
  `moc`. Default value is the value of `qt.moc.auto_preprocessor` (which is
  `true` by default).

* `qt.moc.auto_sys_hdr_dirs`

  If `true`, automatically pass the C++ compiler's system header directories
  to `moc`. Default value is the value of `qt.moc.auto_preprocessor` (which is
  `true` by default).

* `qt.moc.include_with_quotes`

  If `true`, `moc` header outputs will include their source headers with
  quotes (`""`) instead of brackets (`<>`).


### `moc` target types

```
moc{}: cxx_inc{}
automoc{}: target{}
```

* `moc{}`

  The `moc{}` target type represents a C++ source file generated by `moc` from
  a `cxx{}` input and that is expected to be included into another source
  file. It has the default extension `.moc` which can be customized with the
  `extension` target-specific variable.

* `automoc{}`

  The `moc{}` target type represents a group of `cxx{moc_*}` and/or `moc{}`
  members. The set of members is determined by scanning the `hxx{}` and
  `cxx{}` prerequisites of the `automoc{}` target for Qt meta-object macros.
  The `moc` compiler is then run on those prerequisites that matched and the
  resulting targets are added as members to the group. See below for usage
  details.


### Using `moc` with `automoc{}`

The `automoc{}` target type and the corresponding rule allow us to handle
`moc` compilation without having to manually keep track of which source
files must be compiled. For example:

```
exe{hello}: {hxx cxx}{** -moc_*} automoc{hello}
automoc{hello}: {hxx cxx}{** -moc_*}
```

Here the `automoc` rule will scan all the `hxx{}` and `cxx{}` files (because
they are in the `automoc{}` target's prerequisites list) for Qt meta-object
macros and run `moc` on all of those that matched. Note that no explicit
references to any `moc` output targets need to be made anywhere.

The synthesized targets will be named and incorporated into the build
according to the Qt conventions described in the "Compiling C++ source files
with `moc`", "Compiling C++ header files with `moc`", and "Consuming `moc`
outputs" sections below.


### Using `moc` without `automoc{}`

It is also possible to handle `moc` compilation without using the `automoc`
functionality described above. In this case we have to explicitly declare
every dependency involving `moc` outputs:

- The dependency of the primary target on each `moc` output.

- The dependency of each `moc` output on its input file.

The exact setup varies depending on whether we are compiling a source or
header file and each case is described in the following sections.


#### Compiling source files with `moc`

According to the convention, compiling `foo.cxx` with `moc` should produce
`foo.moc` (however the extension can be changed using the `extension`
target-specific variable). The dependency declaration would look like this:

```
moc{foo}: cxx{foo}
```

Note that in this case the generated file is normally included at the end of
`foo.cxx`.


#### Compiling header files with `moc`

According to the convention, compiling `foo.hxx` with `moc` should produce
`moc_foo.cxx` (however the prefix can be changed as discussed below). The
dependency declaration would look like this:

```
cxx{moc_foo}: hxx{foo}
```

Because the target is of type `cxx{}` (as opposed to a dedicated target type
like `moc{}` above), the `moc` compile rule has to rely on this naming
convention to identify `cxx{}` targets that need to be compiled with `moc`.
While it is possible to use other naming schemes, in such cases an explicit
rule hint is necessary:

```
[rule_hint=qt.moc] cxx{foo_bar}: hxx{foo}
```


#### Combined (header and source files) example

The following example shows a header and a source file that will be compiled
with `moc` and their outputs incorporated into an executable (see *Consuming
`moc` outputs* for details):

```
exe{hello}: {hxx cxx}{hello}          # Primary source files.
exe{hello}: cxx{moc_hello} moc{hello} # Moc outputs.

cxx{moc_hello}: hxx{hello} # Compile hello.hxx with moc.
moc{hello}:     cxx{hello} # Compile hello.cxx with moc.
```

Although this example looks relatively simple it quickly becomes tedious and
error-prone to maintain these dependencies as the size of the project grows
and the number of files that have to be compiled by `moc` increases. This is
where the `automoc` functionality can be helpful.


#### Building `moc` outputs (compilation vs. inclusion)

##### Building `cxx{moc_*}`

Normally, output of compiling a header with `moc` is compiled separately as a
standalone translation unit.

However, it is also possible to include it into another translation unit,
similar to the `moc{*}` case below. A potential complication of inclusion is
that it can result in one generated file depending on another generated
file. For example, if `moc_foo.cxx` is included by `foo.cxx`, it makes
`foo.cxx` depend on `moc_foo.cxx`. If, in addition, `foo.cxx` itself needs to
compiled with `moc`, then `foo.moc` ends up depending on `moc_foo.cxx`
transitively via `foo.cxx`.

Handling this case requires the addition of the following explicit dependency
declaration:

```
moc{foo}: cxx{moc_foo}
```

##### Building `moc{*}`

By convention `moc{}` outputs (produced from a C++ source file) are included,
typically into the same source file (so `foo.cxx` in case of `foo.moc`).

Note that this is really the only sensible option because if `foo.moc` were
compiled separately, it would have to include `foo.cxx` in order to bring in
the declarations of the things it's implementing. But then `foo.cxx` could no
longer be compiled otherwise its definitions would be included in multiple
translation units.


### C++ compiler predefs header

It's common practice to pass a header containing the compiler's pre-defined
macros to `moc`. By default the `moc` module will automatically generate this
header and pass it to `moc` but it is possible to disable the automatic mode
if finer-grained control is required. The following example demonstrates both
approaches in one build:

```
# Disable the automatic predefs header for everything by default.
#
qt.moc.auto_predefs = false

# Manual predefs: generate a custom compiler predefs header using
# the cxx module.
#
[rule_hint=cxx.predefs] hxx{moc_predefs}:
{
  # Define a custom macro in the predefs header.
  #
  cxx.coptions += -DCUSTOM_PREDEF_MACRO
}

# Manual predefs: for cxx{moc_foo}, arrange for hxx{moc_predefs} to be
# parsed as a #include before the input header (using moc's --include
# option).
#
cxx{moc_foo}: hxx{moc_predefs}: include = adhoc
cxx{moc_foo}: qt.moc.options += --include $out_base/moc_predefs.hxx

# Automatic predefs: For cxx{moc_bar}, have the moc module automatically
# generate a compiler predefs header (a different one than hxx{moc_predefs}
# above) and pass it to moc.
#
cxx{moc_bar}: qt.moc.auto_predefs = true

```

## `rcc` module

The `rcc` module runs `rcc` (the Qt Resource Compiler) on Qt Resource
Collection files (`.qrc`) producing either a C++ source file or an binary file
("external resource").

Loading the `rcc` module:

```
using qt.rcc
```

### `rcc` configuration variables

```
[strings] qt.rcc.options ?= [null]
```

* `qt.rcc.options`

  Options that will be passed directly to `rcc`. Default value is `null`.

### `rcc` target types

```
qrc{}: file
```

* `qrc{}`

  The `qrc{}` target type represents a Qt Resource Collection file, the input
  file type of the `rcc` program. It has the `.qrc` file extension.


### Compiling resource collection files with `rcc`

According to convention, compiling `qrc{foo}` with `rcc` into C++ source code
(that embeds the declared resources) should produce a file named
`cxx{qrc_foo}`, but any naming scheme can be used.

In the following example `hello.qrc` declares the resources `foo.png` and
`bar.png`:

```
<RCC>
    <qresource prefix="/">
        <file>foo.png</file>
        <file>bar.png</file>
    </qresource>
</RCC>
```

And, in the `buildfile`:

```
exe{hello}: {hxx cxx}{hello} # Primary source files.

exe{hello}: cxx{qrc_hello} # Rcc outputs.

# Compile hello.qrc with rcc to produce a C++ source file that embeds
# the declared resources foo.png and bar.png.
#
cxx{qrc_hello}: qrc{hello}
```

Note that the `rcc` module will track changes to the resources declared in
`hello.qrc` -- `foo.png` and `bar.png` in this case -- and re-run `rcc`
whenever any of them are modified.


### Generated resources

Generated resources must be declared as static prerequisites of the output
target otherwise it is not possible to ensure they exist/up-to-date before
`rcc` is run.

In the following example `hello.qrc` declares the static resources `foo.png`
and `bar.png` and the generated resource `baz.txt`:

```
<RCC>
    <qresource prefix="/">
        <file>foo.png</file>
        <file>bar.png</file>
        <file>baz.txt</file>
    </qresource>
</RCC>
```

And, in the `buildfile`:

```
# Generate baz.txt from baz.txt.in.
#
file{baz.txt}: in{baz}

# Compile hello.qrc with rcc to produce a C++ source file that embeds
# the declared resources foo.png, bar.png, and baz.txt. Note that the
# generated resource baz.txt has to be declared as a static prerequisite.
#
cxx{qrc_hello}: qrc{hello} file{baz.txt}

```

### External resources (binary resource files)

According to convention, compiling `qrc{foo}` with `rcc` into an external
resource file should produce a file named `foo.rcc`. Binary output is selected
by passing the `--binary` option to `rcc`.

```
file{hello.rcc}: qrc{hello}
{
  qt.rcc.options = --binary
}
```

External resources should usually be `posthoc` prerequisites because they are
loaded at runtime and therefore don't need to be updated before the primary
target:

```
exe{hello}: file{hello.rcc}: include = posthoc
```

## `uic` module

Th `uic` module runs `uic` (the Qt User Interface Compiler) on Qt user
interface definition files (`.ui`) to produce a C++ header file implementing
the user interface.

Loading the `uic` module:

```
using qt.uic
```

### `uic` configuration variables

```
[strings] qt.uic.options ?= [null]
```

* `qt.uic.options`

  Options that will be passed directly to `uic`. Default value is `null`.

### `uic` target types

```
ui{}: file
```

* `ui{}`

  The `ui{}` target type represents a Qt User Interface Definition file, the
  input file type of the `uic` program. It has the `.ui` file extension.


### Compiling user interface definition files with `uic`

According to convention, compiling `ui{foo}` with `uic` should produce a file
named `hxx{ui_foo}`, but any naming scheme can be used.

```
exe{hello}: {hxx cxx}{hello} # Primary source files.

exe{hello}: hxx{ui_hello} # Uic output.

# Compile hello.ui with uic to produce a C++ header file.
#
hxx{ui_hello}: ui{hello}
```
